<?php

namespace Application\Module\System\Models;

use Application\Module\Enterprise\Service\AgentsTeamVisibilityList;
use Application\Module\Helpdesk\Service\TicketsVisibility;
use Application\Module\System\Behavior\Team\UpdateUsersResponsibility;
use Application\Module\System\Models\DbTable\Teams;
use Application\Module\System\Services\CannedResponsesCache;
use CannedResponses;
use Macros;
use Singular_Event;
use System_Model_User;
use Useresponse\Acl\Permission;
use View;
use System_Service_UserViews;
use Useresponse\Features\Features;

/**
 * logs row class
 *
 * @property int    $id
 * @property string $type
 * @property string $name
 * @property int    $ticket_visibility
 * @property int    $mailbox_id
 * @property int    $bh_profile_id
 * @property int    $sla_rule_id
 * @property int    $password_policy_id
 * @property string $permissions
 * @property string $options
 * @property int    $is_default
 * @property int    $order
 */
class Team extends \System_Model_Row
{
    const TYPE_GUEST = 'guest';
    const TYPE_USER  = 'user';
    const TYPE_AGENT = 'agent';
    const TYPE_ADMIN = 'admin';

    protected $_tableClass = Teams::class;

    protected $resetUsersOnSave = false;

    public function objectsReport()
    {
        $view          = new \System_Model_View();
        $view->type    = 'object';
        $view->options = serialize([
            'mandatory' => [
                'state' => [
                    [
                        'equality' => 'is',
                        'value'    => \System_Mapper_State::NORMAL
                    ],
                    [
                        'equality' => 'is',
                        'value'    => \System_Mapper_State::ARCHIVED
                    ]
                ]
            ],
            'variable' => [
                [
                    'created_by' => [
                        [
                            'equality' => 'selected-team',
                            'type'     => 'is',
                            'value'    => [$this->id],
                        ]
                    ],
                ]
            ]
        ]);

        return $view;
    }

    public function option(string $name, $default = null)
    {
        $value = $default;

        if ($this->options) {
            $options = json_decode($this->options, true);

            if (array_key_exists($name, $options)) {
                $value = $options[$name];
            }
        }

        return $value;
    }

    public function setOptions(string $name, $value)
    {
        $options        = $this->options ? json_decode($this->options, true) : [];
        $options[$name] = $value;
        $this->options  = json_encode($options);
    }

    public function isGuest(): bool
    {
        return $this->type === self::TYPE_GUEST;
    }

    public function isAdmin(): bool
    {
        return $this->type === self::TYPE_ADMIN;
    }

    public function isAgent(): bool
    {
        return $this->type === self::TYPE_AGENT;
    }

    public function isUser(): bool
    {
        return $this->type === self::TYPE_USER;
    }

    public function isDefault(): bool
    {
        return (int)$this->is_default > 0;
    }

    public function countUsers()
    {
        return (int)\db()->fetchOne(
           \db()->select()
                ->from(\tableName('users'))
                ->reset(\Zend_Db_Select::COLUMNS)
                ->columns(['count(*)'])
                ->where('team_id = ?', $this->id)
                ->where('state IN (?)', [\User::STATE_NORMAL, \User::STATE_ANONYMOUS])
        );
    }

    public function getReportUrl()
    {
        $reportOptions = [
            'variable' => [
                [
                    'team' => [[
                        'equality' => 'is',
                        'value'    => (int)$this->id
                    ]],
                    'state' => [[
                        'equality' => 'is_not',
                        'value'    => System_Model_User::STATE_ANONYMOUS
                    ]]
                ]
            ]
        ];

        $report = new View();
        $report->type = 'user';
        System_Service_UserViews::forceApplyOptions($report, ['options' => $reportOptions]);

        return $report->getOptionsUrl();
    }

    public function addPermission(string $permission, bool $allowed = true): self
    {
        $permissions = json_decode($this->permissions);
        $permissions->{$permission} = $allowed;
        $this->permissions = json_encode($permissions);

        return $this;
    }

    public function hasPermission(string $permission): bool
    {
        $permissions = json_decode($this->permissions);

        return isset($permissions->{$permission});
    }

    public function isAvailableFor(\User $user): bool
    {
        return Features::isOff(Features::FEATURE_TICKETS_VISIBILITY) ||
            TicketsVisibility::hasAccessToGroup($user, $this);
    }

    public function hasAnyPermission(): bool
    {
        $permissions = json_decode($this->permissions, true);
        foreach ($permissions as $value) {
            if ($value) {
                return true;
            }
        }

        return false;
    }

    public function hasAgentAccess(): bool
    {
        return $this->type === self::TYPE_AGENT || $this->type === self::TYPE_ADMIN;
    }

    public function transform()
    {
        return [
            'id'   => (int)$this->id,
            'name' => $this->name,
        ];
    }

    public function setPostData(array $data, \User $user)
    {
        $this->name = $data['name'] ?? null;

        if (empty($this->name)) {
            throw new \Exception('Name is required');
        }

        $this->type = $data['type'] ?? null;

        if (!in_array($this->type, [self::TYPE_USER, self::TYPE_AGENT, self::TYPE_ADMIN], true)) {
            throw new \Exception('Invalid team type');
        }

        $permissions = new \stdClass();
        if (!empty($data['permissions'])) {
            foreach ($data['permissions'] as $permission => $allowed) {
                $allowed = $allowed && $allowed !== 'false';

                if (!$this->isAdmin() || ((stripos($permission, 'can_access_forum_') === 0 || stripos($permission, 'can_manage_forum') === 0) && !$allowed)) {
                    $permissions->{$permission} = $allowed && $allowed;
                }
            }
        }

        $this->permissions = json_encode($permissions);

        foreach (['mailbox_id', 'sla_rule_id', 'ticket_visibility', 'password_policy_id', 'bh_profile_id'] as $field) {
            if (array_key_exists($field, $data)) {
                $this->{$field} = (int)$data[$field];
            }
        }

        if ($this->isUser()) {
            $this->ticket_visibility = 0;
            $this->mailbox_id = 0;
        } else {
            $this->sla_rule_id = 0;
        }

        $options = new \stdClass();
        if (module('helpdesk')->active() && (int)$this->ticket_visibility === TicketsVisibility::PRIVILEGE_IN_GROUPS) {
            $options->teams = !empty($data['teams']) ? array_map('intval', $data['teams']) : [];
            sort($options->teams);
        }

        $this->options = json_encode($options);

        $this->resetUsersOnSave = !empty($data['reset_users']);

        Singular_Event::dispatch('agent.teams.manage.post', $this, $data, $user);
    }

    public function canDelete(\User $user): bool
    {
        return !$this->isNew() && !$this->isDefault() && $this->canManage($user);
    }

    public function canManage(\User $user): bool
    {
        return $user->canManagePeople() && ($this->isUser() || $user->role()->isAdmin());
    }

    public function modifiedUserIds(): array
    {
        if ($this->isNew()) {
            return [];
        }

        return \db()->fetchCol(
            \db()->select()
                ->from(['users' => \tableName('users')])
                ->reset(\Zend_Db_Select::COLUMNS)
                ->columns(['users.id'])
                ->joinLeft(
                    ['settings' => \tableName('user_settings')],
                    'settings.user_id = users.id AND
                    settings.key IN ("team_type", "permissions", "ticket_visibility", "team_options")',
                    []
                )
                ->where('users.team_id = ?', $this->id)
                ->where('settings.id IS NOT NULL')
                ->group('users.id')
        );
    }

    public function userIds(): array
    {
        if ($this->isNew()) {
            return [];
        }

        return \db()->fetchCol(
            \db()->select()
                ->from(['users' => \tableName('users')])
                ->reset(\Zend_Db_Select::COLUMNS)
                ->columns(['users.id'])
                ->where('users.team_id = ?', $this->id)
                ->group('users.id')
        );
    }

    protected function _insert()
    {
        $this->order = Teams::nextOrder();
        parent::_insert();
    }

    protected function _postDelete()
    {
        $setType = $this->hasAgentAccess() ? self::TYPE_AGENT : self::TYPE_USER;
        (new \Users())->update(['team_id' => Teams::defaultIdOfType($setType)], sprintf('team_id = %d', $this->id));
        (new \Objects())->update(['team_id' => 0], sprintf('team_id = %d', $this->id));

        (new CannedResponses())->update(
            ['available_type' => 'all', 'available_value' => 0],
            sprintf('available_type = "selected_team" AND available_value = %d', $this->id)
        );

        (new Macros())->update(
            ['available_type' => 'all', 'available_value' => 0],
            sprintf('available_type = "selected_team" AND available_value = %d', $this->id)
        );

        CannedResponsesCache::update();

        parent::_postDelete();
    }

    protected function _postUpdate()
    {
        parent::_postUpdate();

        (new UpdateUsersResponsibility($this))
            ->withCondition(function () {
                return $this->isModified('ticket_visibility') ||
                    ($this->isModified('type') && !$this->isAdmin() && !$this->isAgent());
            })
            ->provoke();
        if (Features::isOn(Features::FEATURE_TICKETS_VISIBILITY)) {
            AgentsTeamVisibilityList::update();
        }
    }

    protected function written()
    {
        parent::written();

        CannedResponsesCache::update();

        if ($this->resetUsersOnSave) {
            $this->resetUsersOnSave = false;

            $userIds = $this->modifiedUserIds();
            if (($key = array_search(loggedUser()->id, $userIds)) !== false) {
                unset($userIds[$key]);
            }

            if (count($userIds) > 0) {
                (new \UserSettings())->delete(
                    sprintf(
                        '`key` IN ("team_type", "permissions", "ticket_visibility", "team_options") AND user_id IN (%s)',
                        implode(',', $userIds)
                    )
                );
            }
        }
    }

    protected function _postInsert()
    {
        if (
            isset($this->ticket_visibility) &&
            (int)$this->ticket_visibility === TicketsVisibility::PRIVILEGE_IN_GROUPS
        ) {
            $teams = $this->option('teams');
            $teams[] = (int)$this->id;
            $this->setOptions('teams', $teams);
            \db()->update(\tableName('teams'), [
                'options' => $this->options
            ], \db()->quoteInto('id = ?', $this->id));
        }
        parent::_postInsert();
    }

    public function getUsersCount()
    {
        return \db()->fetchOne(
            \db()->select()
            ->from(\tableName('users'))
            ->reset(\Zend_Db_Select::COLUMNS)
            ->columns("COUNT(*)")
            ->where('team_id = ?', $this->id)
            ->where('state IN (?)', [\User::STATE_NORMAL])
        );
    }

    public function hasOnlineAgents(): bool
    {
        $agents = (new \Users())->findByTeam($this->id);

        $hasWorkAgents = false;
        foreach ($agents as $agent) {
            if ((bool)$agent->settings()->agent_online) {
                $hasWorkAgents = true;
                break;
            }
        }

        return $hasWorkAgents;
    }
}
