<?php

use Application\Module\Enterprise\Models\DbTable\BhProfiles;
use Application\Module\Helpdesk\Service\TicketsVisibility;
use Application\Module\System\Models\DbTable\PasswordPolicies;
use Application\Module\System\Models\DbTable\Teams;
use Application\Module\System\Models\Team;
use Useresponse\Acl\Manager;
use Useresponse\Application\App;
use Useresponse\Features\Features;

class AgentTeamsController extends System_Controller_Abstract_Agent
{
    public function init()
    {
        parent::init();

        if (!$this->isAjax()) {
            $this->view->layout()->setLayout('agent');
        }
    }

    public function listAction()
    {
        $type = $this->_getParam('type', 'all');
        $sortBy = $this->_getParam('sortBy', null);
        $reverseSort = $this->_getParam('reverse', 'false') === 'true';
        $this->view->title = t('Teams');
        $this->view->type  = $type;
        $teams = $type === 'all' ? Teams::findAll() :
            Teams::findByTypes($type === 'agent' ? [Team::TYPE_AGENT, Team::TYPE_ADMIN] : [Team::TYPE_USER]);

        $teamsList = [];
        foreach ($teams as $team) {
            $teamsList[] = $team;
        }
        if (!empty($sortBy)) {
            uasort($teamsList, function (Team $team1, Team $team2) use ($sortBy, $reverseSort) {
                if ($sortBy === 'name') {
                    return $reverseSort ? strcmp($team2->name, $team1->name) : strcmp($team1->name, $team2->name);
                }

                if ($sortBy === 'users_count') {
                    return $reverseSort ? $team2->getUsersCount() < $team1->getUsersCount() : $team1->getUsersCount() < $team2->getUsersCount();
                }

                return true;
            });
        }
        $this->view->teams = $teamsList;
        $this->view->sortBy = $sortBy;
        $this->view->reverseSort = $reverseSort;

        $teamsListBlock = $this->view->render('views/lists/teams.phtml');

        if ($this->isAjax()) {
            $this->autoresponder->setOutput($teamsListBlock);
        } else {
            $this->view->assign([
                'viewList'   => $teamsListBlock,
                'customMenu' => $this->view->render('users-manage-popups/reports-popup.phtml'),
            ]);
        }
    }

    public function manageAction()
    {
        $teamId = (int)$this->_getParam('team_id');
        $team   = $teamId > 0 ? Teams::findById($teamId) : new Team();

        if (!$team instanceof Team) {
            $this->notFound();
        }

        $features = (object)[
            'policies' => Features::isOn(Features::FEATURE_SECURITY_POLICY),
            'tickets'  => Features::isOn(Features::FEATURE_TICKETS_VISIBILITY),
            'helpdesk' => module('helpdesk')->active(),
            'bh'       => Features::isOn(Features::FEATURE_BUSINESS_HOURS)&& settings('business_hours_enabled')->isOn(),
            'forums'   => Features::isOn(Features::FEATURE_FORUMS) && Forums_Service_Forums::isMulti(),
            'sla'      => Features::isOn(Features::FEATURE_SLA) &&
                App::services()->get('sla')->activeModule() &&
                module(App::services()->get('sla')->activeModule())->active(),
        ];

        $teamRowSet = Teams::findAll();
        $allTeams   = [];

        foreach ($teamRowSet as $teamRow) {
            if ($teamRow->isAdmin() || $teamRow->isAgent()) {
                $allTeams[] = $teamRow;
            }
        }

        $allTeams[] = (object)['id' => 0, 'name' => 'No Team'];

        $teamCols = array_chunk($allTeams, ceil(count($allTeams) / 2));
        if (empty($teamCols[1])) {
            $teamCols[1] = [];
        }

        $permissions = Manager::permissions();

        $this->view->assign([
            'title'       => t($team->isNew() ? 'Add Team' : 'Edit Team'),
            'permissions' => $permissions,
            'features'    => $features,
            'teamCols'    => $teamCols,
            'mailboxes'   => (new \Mailboxes())->outgoingMailboxes(),
            'policies'    => $features->policies ? PasswordPolicies::findAll() : null,
            'bhProfiles'  => $features->bh ? BhProfiles::findCustom() : null,
            'slaRules'    => $features->sla ? (new \SlaRules())->fetchAll() : null,
            'isNew'       => $team->isNew(),
            'canDelete'   => $team->canDelete($this->loggedUser),
        ]);

        if ($team->isDefault()) {
            $teamPermissions = [];
            $updatedPermissions = json_decode($team->permissions);
            foreach ($permissions as $permission) {
                if ($permission->slug() === 'agent') {
                    foreach ($permission->dependencies() as $perm) {
                        if (isset($updatedPermissions->{$perm->slug()})) {
                            $teamPermissions[$perm->slug()] = $updatedPermissions->{$perm->slug()};
                        }
                        $teamPermissions[$perm->slug()] = $team->isAdmin() ? true : ($team->isUser() ? $perm->isAllowedByDefault() : $perm->isAllowedForAgentByDefault());
                    }
                    continue;
                }
                if (isset($updatedPermissions->{$permission->slug()})) {
                    $teamPermissions[$permission->slug()] = $updatedPermissions->{$permission->slug()};
                    continue;
                }
                $teamPermissions[$permission->slug()] = $team->isAdmin() ? true : ($team->isUser() ? $permission->isAllowedByDefault() : $permission->isAllowedForAgentByDefault());
            }
            if ($team->isUser()) {
                $teamPermissions = array_merge($teamPermissions, json_decode($team->permissions, true));
            }
        } else {
            $teamPermissions = json_decode($team->permissions);
        }

        $teamData = new ArrayObject([
            'id'                  => (int)$team->id,
            'mailbox_id'          => (int)$team->mailbox_id,
            'sla_rule_id'         => (int)$team->sla_rule_id,
            'password_policy_id'  => (int)$team->password_policy_id,
            'ticket_visibility'   => module('helpdesk')->active() && ($team->isNew() || $team->isUser())
                ? TicketsVisibility::PRIVILEGE_ALL_TICKETS
                : (int)$team->ticket_visibility,
            'type'                => $team->type,
            'name'                => $team->name,
            'permissions'         => $teamPermissions,
            'options'             => json_decode($team->options),
            'is_new'              => $team->isNew(),
            'features'            => $features,
            'save_url'            => view()->url(['team_id' => (int)$team->id], 'agent_team_save'),
            'delete_url'          => view()->url(['team_id' => (int)$team->id], 'agent_team_delete'),
            'visibility_in_teams' => module('helpdesk')->active() ? TicketsVisibility::PRIVILEGE_IN_GROUPS : 0,
            'has_modified_users'  => count($team->modifiedUserIds()) > 0,
            'is_default'          => $team->isDefault(),
            'forums_private_list' => Forums::privateAssoc(),
        ]);

        Singular_Event::dispatch('agent.teams.manage.action', $this, $team, $teamData);

        $viewList = $this->view->render('views/manage/team.phtml');

        if ($this->isAjax()) {
            $this->autoresponder->setParam('teamData', $teamData->getArrayCopy())->setOutput($viewList);
        } else {
            Singular_AppClient::setParam('teamData', $teamData->getArrayCopy());
            $this->view->assign('customMenu', $this->view->render('users-manage-popups/reports-popup.phtml'));
            $this->view->assign('viewList', $viewList);
        }
    }

    public function saveAction()
    {
        if (!$this->getRequest()->isPost() || !$this->isAjax()) {
            $this->invalidRequest();
        }

        $teamId = (int)$this->_getParam('team_id');
        $team   = $teamId > 0 ? Teams::findById($teamId) : new Team();

        if (!$team instanceof Team) {
            $this->notFound();
        }

        try {
            $team->setPostData($this->getRequest()->getPost(), $this->loggedUser);
            $team->save();
            $this->autoresponder->setParam('teamUrl', view()->url(['team_id' => $team->id], 'agent_team'))
                ->setParam('manageList', $this->fetchTemplate('users-manage-popups/teams.phtml'))
                ->setSuccess($teamId > 0 ? 'Team is updated!' : 'New team is added!');
        } catch (Exception $e) {
            $this->autoresponder->setError($e->getMessage());
        }
    }

    public function deleteAction()
    {
        if (!$this->isAjax()) {
            $this->invalidRequest();
        }

        $teamId = (int)$this->_getParam('team_id');
        $team   = $teamId > 0 ? Teams::findById($teamId) : new Team();

        if (!$team instanceof Team) {
            $this->notFound();
        }

        if (!$team->canDelete($this->loggedUser)) {
            $this->accessDenied();
        }

        $team->delete();
        $this->autoresponder->setSuccess('Team is deleted!');
    }

    public function notesAction()
    {
        if (!$this->isAjax()) {
            $this->invalidRequest();
        }

        $teamId = $this->_getParam('team_id');
        $userId = $this->_getParam('user_id', 'all');
        $users  = (new Users())->findByTeam($teamId);
        $team   = Teams::findById($teamId);

        if ($userId === 'all') {
            $userIds = [];
            foreach ($users as $user) {
                $userIds[] = $user->id;
            }
            $notes = empty($userIds) ? [] : Singular_Loader::get('Comments')->findNotesForUsers($userIds);
        } else {
            $notes = Singular_Loader::get('Comments')->findNotesForUsers([$userId]);
        }

        $this->view->assign([
            'notes'                => $notes,
            'activeObjectId'       => $this->_getParam('activeObjectId'),
            'activeObjectTabTitle' => $this->_getParam('activeObjectTabTitle'),
            'team'                 => $team,
            'users'                => $users,
            'userId'               => $userId,
        ]);

        $this->autoresponder->setOutput($this->fetchTemplate('views/lists/team-notes.phtml'));
    }

    public function userEditAction()
    {
        if (!$this->isAjax()) {
            $this->invalidRequest();
        }

        $userId = (int)$this->_getParam('user_id');
        $user   = $userId > 0 ? (new Users())->findById($userId) : null;

        if (!$user instanceof User) {
            $this->autoresponder->setError('Not found');
            return;
        }

        if (!$this->loggedUser->canManagePermissions($user)) {
            $this->autoresponder->setError('Access denied');
            return;
        }

        $role           = $user->role();
        $team           = $user->team();
        $allPermissions = Manager::permissions();
        $permissions    = [];

        $default = (object)[
            'team_type'         => $team->type,
            'ticket_visibility' => (int)$team->ticket_visibility,
            'teams'             => $team->option('teams', []),
            'permissions'       => [],
        ];

        foreach ($allPermissions as $permission) {
            if ($permission->slug() === 'agent') {
                foreach ($permission->dependencies() as $dependency) {
                    $permissions[$dependency->slug()] = $role->allowed($dependency->slug());
                    $default->permissions[$dependency->slug()] = $dependency->defaultValue($team);
                }
            } else {
                $permissions[$permission->slug()] = $role->allowed($permission->slug());
                $default->permissions[$permission->slug()] = $permission->defaultValue($team);
            }
        }

        foreach (json_decode($team->permissions, true) ?? [] as $permission => $allowed) {
            $default->permissions[$permission] = $allowed;
        }

        $teamRowSet = Teams::findAll();
        $allTeams   = [];

        foreach ($teamRowSet as $teamRow) {
            if ($teamRow->isAdmin() || $teamRow->isAgent()) {
                $allTeams[] = $teamRow;
            }
        }

        $allTeams[] = (object)['id' => 0, 'name' => 'No Team'];

        $teamCols = array_chunk($allTeams, ceil(count($allTeams) / 2));
        if (empty($teamCols[1])) {
            $teamCols[1] = [];
        }

        $this->view->assign([
            'team_type'         => $role->type(),
            'ticket_visibility' => $role->ticketVisibility(),
            'teams'             => $role->option('teams', []),
            'teamCols'          => $teamCols,
            'permissions'       => $permissions,
            'allPermissions'    => $allPermissions,
            'hasForums'         => Features::isOn(Features::FEATURE_FORUMS) && Forums_Service_Forums::isMulti(),
            'hasHelpDesk'       => module('helpdesk')->active(),
            'hasVisibility'     => Features::isOn(Features::FEATURE_TICKETS_VISIBILITY),
            'saveUrl'           => $this->view->url(['user_id' => $user->id], 'agent_user_permissions_save'),
            'defaultOptions'    => $default,
            'isModified'        => $user->hasModifiedPermissions(),
            'isMe'              => (int)$user->id === (int)$this->loggedUser->id,
            'hasChat'           => module('helpdesk')->active(),
        ]);

        $this->autoresponder->setOutput($this->fetchTemplate('system_agent-teams_user-edit.phtml'));
    }

    public function userSaveAction()
    {
        if (!$this->isAjax() || !$this->getRequest()->isPost()) {
            $this->invalidRequest();
        }

        $userId = (int)$this->_getParam('user_id');
        $user   = $userId > 0 ? (new Users())->findById($userId) : null;

        if (!$user instanceof User) {
            $this->autoresponder->setError('Not found');

            return;
        }

        if (!$this->loggedUser->canManagePeople($user)) {
            $this->autoresponder->setError('Access denied');

            return;
        }

        try {
            $user->saveTeamData($this->getRequest()->getPost());
            $this->autoresponder->setSuccess('Profile is updated!')
                ->setParam('user_id', $user->id)
                ->setParam('has_modified_permissions', $user->hasModifiedPermissions());
        } catch (Exception $e) {
            $this->autoresponder->setError($e->getMessage());
        }
    }
}
